terraform {
  required_providers {
    coder = {
      source  = "coder/coder"
      version = "2.1.2"
    }
    docker = {
      source  = "kreuzwerker/docker"
      version = "~> 3.0.2"
    }
  }
}

# Admin parameters

locals {
  username = data.coder_workspace_owner.me.name
}

data "coder_provisioner" "me" {
}

provider "docker" {
  host = "unix:///var/run/docker.sock"
}

data "coder_workspace" "me" {
}

data "coder_workspace_owner" "me" {
}

data "coder_external_auth" "gitea" {
  id = "gitea"
}

data "coder_parameter" "force_rebuild" {
  name         = "force_rebuild"
  type         = "bool"
  description  = "Rebuild the Docker image rather than use the cached one."
  mutable      = true
  default      = false
  ephemeral    = true
}

module "code-server" {
  source          = "registry.coder.com/modules/code-server/coder"
  version         = "1.0.28"
  agent_id        = coder_agent.main.id
  display_name    = "VS Code"
  slug            = "code"
  install_version = "4.96.4"
  folder          = "/home/${local.username}/workspace"
  port            = "8080"
  extensions      = [
    "eamodio.gitlens",
    "PKief.material-icon-theme"
  ]
}

module "personalize" {
  source   = "registry.coder.com/modules/personalize/coder"
  version  = "1.0.28"
  agent_id = coder_agent.main.id
  path     = "~/personalize.sh"
}

module "nodejs" {
  source               = "registry.coder.com/modules/nodejs/coder"
  version              = "1.0.28"
  agent_id             = coder_agent.main.id
  nvm_install_prefix   = "/home/${local.username}/.nvm"
  node_versions        = [
    "20",
    "22"
  ]
  default_node_version = "22"
}

module "coder-login" {
  source   = "registry.coder.com/modules/coder-login/coder"
  version  = "1.0.28"
  agent_id = coder_agent.main.id
}

module "git-config" {
  source                = "registry.coder.com/modules/git-config/coder"
  version               = "1.0.28"
  agent_id              = coder_agent.main.id
  allow_username_change = false
}

module "git-commit-signing" {
  source   = "registry.coder.com/modules/git-commit-signing/coder"
  version  = "1.0.28"
  agent_id = coder_agent.main.id
}

provider "coder" {
}

resource "coder_agent" "main" {
  arch = data.coder_provisioner.me.arch
  os   = "linux"
  dir  = "/home/${local.username}/workspace/"
}

resource "coder_script" "auto-update" {
  agent_id           = coder_agent.main.id
  script             = "/opt/scripts/auto-update.sh"
  display_name       = "Auto Update"
  run_on_start       = true
  start_blocks_login = false
}

resource "docker_volume" "home_volume" {
  name = "coder-${data.coder_workspace.me.id}-home"
  # Protect the volume from being deleted due to changes in attributes.
  lifecycle {
    ignore_changes = all
  }
  # Add labels in Docker to keep track of orphan resources.
  labels {
    label = "coder.owner"
    value = data.coder_workspace_owner.me.name
  }
  labels {
    label = "coder.owner_id"
    value = data.coder_workspace_owner.me.id
  }
  labels {
    label = "coder.workspace_id"
    value = data.coder_workspace.me.id
  }
  # This field becomes outdated if the workspace is renamed but can
  # be useful for debugging or cleaning out dangling volumes.
  labels {
    label = "coder.workspace_name_at_creation"
    value = data.coder_workspace.me.name
  }
}

resource "docker_image" "main" {
  name = "coder-${data.coder_workspace.me.id}"
  build {
    context = "."
    build_args = {
      USER = local.username
    }
  }
  triggers = {
    docker_sha1 = sha1(filesha1("./Dockerfile"))
    terraform_sha1 = sha1(filesha1("./Base.tf"))
    rebuild_sha1 = data.coder_parameter.force_rebuild.value
    assets_sha1 = sha1(join("", [for f in fileset(path.module, "./assets/*") : filesha1(f)]))
  }
}

resource "docker_container" "workspace" {
  count = data.coder_workspace.me.start_count
  image = docker_image.main.name
  # Uses lower() to avoid Docker restriction on container names.
  name = "coder-${data.coder_workspace_owner.me.name}-${lower(data.coder_workspace.me.name)}"
  # Hostname makes the shell more user friendly: coder@my-workspace:~$
  hostname = data.coder_workspace.me.name
  # Use the docker gateway if the access URL is 127.0.0.1
  entrypoint = ["sh", "-c", replace(coder_agent.main.init_script, "/localhost|127\\.0\\.0\\.1/", "host.docker.internal")]
  env        = ["CODER_AGENT_TOKEN=${coder_agent.main.token}"]
  host {
    host = "host.docker.internal"
    ip   = "host-gateway"
  }
  volumes {
    container_path = "/home/${local.username}"
    volume_name    = docker_volume.home_volume.name
    read_only      = false
  }
  # Add labels in Docker to keep track of orphan resources.
  labels {
    label = "coder.owner"
    value = data.coder_workspace_owner.me.name
  }
  labels {
    label = "coder.owner_id"
    value = data.coder_workspace_owner.me.id
  }
  labels {
    label = "coder.workspace_id"
    value = data.coder_workspace.me.id
  }
  labels {
    label = "coder.workspace_name"
    value = data.coder_workspace.me.name
  }
}
